﻿using Microsoft.Crm.Sdk.Messages;
using Microsoft.Xrm.Sdk;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading.Tasks;
using VA.PPMS.Context;
using VA.PPMS.Context.Interface;
using VA.PPMS.IWS.MappingService.Helpers;
using VA.PPMS.ProviderData;

namespace VA.PPMS.IWS.MappingService.Mappers
{
    public class MapOtherIdentifierToCrm : MapperRelatedBase
    {
        public MapOtherIdentifierToCrm(IPpmsContextHelper ppmsContextHelper, IPpmsHelper ppmsHelper)
            : base(ppmsContextHelper, ppmsHelper)
        {}

        public async Task<ppms_otherprovideridentifier> MapInsert(OtherIdentifier identifier, Account provider)
        {
            if (identifier == null) HandleException("Invalid identifier provided.");

            // map properties
            var entity = new ppms_otherprovideridentifier
            {
                Id = Guid.NewGuid(),
                ppms_name = identifier?.Name,
                ppms_state = identifier?.IdentifierState,
                ppms_identifiertype = EnumHelper.MapEnumToOptionSetValue<ppms_otherprovideridentifier_ppms_identifiertype>(identifier?.IdentifierTypeCode.ToString())
            };

            // Set owner to CCN
            if (!ForVaNetwork && Owner != null) entity.OwnerId = Owner;

            await Task.Run(() => {});

            return entity;
        }

        public async Task<Entity> MapUpdate(OtherIdentifier entity, Account provider)
        {
            if (entity == null) HandleException("Invalid identifier provided.");
            
            // ReSharper disable once PossibleNullReferenceException
            var crmEntity = GetCrmEntity(provider, entity.Name);
            
            // matching account not found
            if (crmEntity == null)
            {
                return await MapInsert(entity, provider);
            }

            var newEntity = new ppms_otherprovideridentifier()
            {
                Id = crmEntity.Id,
                ppms_correlationid = crmEntity.ppms_correlationid,
                ppms_name = crmEntity.ppms_name,
                ppms_state = crmEntity.ppms_state,
                ppms_identifiertype = crmEntity.ppms_identifiertype
            };

            // set properties
            // name
            if (IsChanged(entity.Name, newEntity.ppms_name)) newEntity.ppms_name = entity.Name;

            // identifier code
            newEntity.ppms_identifiertype = EnumHelper.MapEnumToOptionSetValue<ppms_otherprovideridentifier_ppms_identifiertype>(entity.IdentifierTypeCode.ToString());

            // identifier state
            if (IsChanged(entity.IdentifierState, newEntity.ppms_state)) newEntity.ppms_state = entity.IdentifierState;

            // Set owner to CCN
            if (!ForVaNetwork && Owner != null) newEntity.OwnerId = Owner;

            // return mapped entity
            return newEntity;
        }

        private static ppms_otherprovideridentifier GetCrmEntity(Account provider, string id)
        {
            var list = provider.ppms_account_ppms_otherprovideridentifier;
            if (list == null) return null;

            var ppmsOtherprovideridentifiers = list as ppms_otherprovideridentifier[] ?? list.ToArray();

            return ppmsOtherprovideridentifiers.Any() ? ppmsOtherprovideridentifiers.FirstOrDefault(x => x.ppms_name == id) : null;
        }

        private static OtherIdentifier ConvertEntity<T>(T entity)
        {
            return (OtherIdentifier)Convert.ChangeType(entity, typeof(OtherIdentifier));
        }

        public override async Task<Entity> MapUpdate<T>(T entity, Entity parent)
        {
            return await MapUpdate(ConvertEntity(entity), (Account)parent);
        }

        public override async Task<Entity> MapInsert<T>(T entity, Entity parent)
        {
            return await MapInsert(ConvertEntity(entity), (Account)parent);
        }

        public override void AddChildrenToProvider(IList<Entity> entities, Entity parent)
        {
            if (IsWithinContext) AssociateRelatedEntities(parent, entities, "ppms_account_ppms_otherprovideridentifier");
            else
            {
                if (entities != null && entities.Count > 0)
                {
                    var account = (Account)parent;
                    if (account != null) account.ppms_account_ppms_otherprovideridentifier = ConvertEntityList<ppms_otherprovideridentifier>(entities);
                }
            }
        }

        public override IEnumerable<SetStateRequest> MapDelete<T>(IList<T> entities, Entity parent)
        {
            if (entities == null || !entities.Any()) return null;

            // Check provider
            var provider = (Account)parent;
            if (provider == null) return null;

            var list = new List<ppms_otherprovideridentifier>();
            var relatedEntities = provider.ppms_account_ppms_otherprovideridentifier.ToList();
            OtherIdentifier entity;

            if (relatedEntities != null && relatedEntities.Any())
            {
                // Map schema entities for delete
                foreach (var item in entities)
                {
                    entity = ConvertEntity(item);
                    var matches = relatedEntities.Where(p => p.ppms_name == entity.Name);
                    list.AddRange(matches);
                }
            }

            return EntityDelete((IEnumerable<ppms_otherprovideridentifier>)list);
        }
    }
}